<?php
namespace App\Http\Controllers\Admin;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Admin\Category;
use App\Models\Sports;
use App\Models\Venue;
use App\Models\Court;
use App\Models\Days;
use App\Models\Slots;
use App\Models\SlotTime;
use App\Models\CutomSlots;
use App\Models\PlayerType;
use App\Models\CutomSlotsTiming;
use App\Models\Coaching;
use App\Models\CoachingTimings;
use App\Models\Booking;
use DataTables;
use Image;
use File;
use Exception;
use Cviebrock\EloquentSluggable\Services\SlugService;
use App\Rules\CouchingTimeSave;
use App\Rules\WeeklySlotTimeSave;

class CourtController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //
        try {
            if ($request->ajax()) {
                $data = Court::with("venue.sports.category")
                    ->select(
                        "id",
                        "court_name",
                        "category_id",
                        "sports_id",
                        "venue_id",
                        "image",
                        "slug",
                        "status"
                    )
                    ->orderBy("id", "DESC")
                    ->get();
                return Datatables::of($data)
                    ->addIndexColumn()
                    ->addColumn("action", function ($row) {
                        $btn =
                            '
                        <a href=' .
                            route("admin.court-time-schedule", $row->id) .
                            '>
                        <button type="button" class="btn btn-icon btn-active-light-primary w-30px h-30px ms-auto" data-bs-toggle="tooltip" data-bs-custom-class="tooltip-inverse" data-bs-placement="top" title="Schedule time">
                        <span class="svg-icon svg-icon-3">
                        <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><polyline points="12 6 12 12 15 15"></polyline></svg>
                        </span>
                        </button></a>
                        
                        
                        <button type="button" class="btn btn-icon btn-active-light-primary w-30px h-30px ms-auto edit_sports" data-id="' .
                            $row->id .
                            '">
                        <span class="svg-icon svg-icon-3">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path opacity="0.3" d="M21.4 8.35303L19.241 10.511L13.485 4.755L15.643 2.59595C16.0248 2.21423 16.5426 1.99988 17.0825 1.99988C17.6224 1.99988 18.1402 2.21423 18.522 2.59595L21.4 5.474C21.7817 5.85581 21.9962 6.37355 21.9962 6.91345C21.9962 7.45335 21.7817 7.97122 21.4 8.35303ZM3.68699 21.932L9.88699 19.865L4.13099 14.109L2.06399 20.309C1.98815 20.5354 1.97703 20.7787 2.03189 21.0111C2.08674 21.2436 2.2054 21.4561 2.37449 21.6248C2.54359 21.7934 2.75641 21.9115 2.989 21.9658C3.22158 22.0201 3.4647 22.0084 3.69099 21.932H3.68699Z" fill="currentColor"></path>
                            <path d="M5.574 21.3L3.692 21.928C3.46591 22.0032 3.22334 22.0141 2.99144 21.9594C2.75954 21.9046 2.54744 21.7864 2.3789 21.6179C2.21036 21.4495 2.09202 21.2375 2.03711 21.0056C1.9822 20.7737 1.99289 20.5312 2.06799 20.3051L2.696 18.422L5.574 21.3ZM4.13499 14.105L9.891 19.861L19.245 10.507L13.489 4.75098L4.13499 14.105Z" fill="currentColor"></path>
                        </svg>
                        </span>
                        </button>
                        <button type="button" class="btn btn-icon btn-active-light-primary w-30px h-30px ms-auto delete_venue" data-id="' .
                            $row->id .
                            '" id="kt_users_delete_two_step">
                        <span class="svg-icon svg-icon-3">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M5 9C5 8.44772 5.44772 8 6 8H18C18.5523 8 19 8.44772 19 9V18C19 19.6569 17.6569 21 16 21H8C6.34315 21 5 19.6569 5 18V9Z" fill="currentColor"></path>
                            <path opacity="0.5" d="M5 5C5 4.44772 5.44772 4 6 4H18C18.5523 4 19 4.44772 19 5V5C19 5.55228 18.5523 6 18 6H6C5.44772 6 5 5.55228 5 5V5Z" fill="currentColor"></path>
                            <path opacity="0.5" d="M9 4C9 3.44772 9.44772 3 10 3H14C14.5523 3 15 3.44772 15 4V4H9V4Z" fill="currentColor"></path>
                        </svg>
                        </span>
                        </button>';
                        return $btn;
                    })
                    ->editColumn("image", function ($row) {
                        if (isset($row->image) && $row->image != "") {
                            return asset($row->image);
                        }
                        return asset("admin/assets/media/no_img.png");
                    })
                    ->rawColumns(["action"])
                    ->make(true);
            }
            $category = Category::all();
            return view("admin.court", compact("category"));
        } catch (Exception $e) {
            // return response()->json(
            //     [
            //         "success" => false,
            //         "message" => "Some issue.Please contact admin",
            //     ],
            //     204
            // );
            echo $e->getMessage();
        }
    }
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }
    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //
        $this->validate(
            $request,
            [
                "file" => "required|image|mimes:jpeg,png,jpg,gif,svg|max:2048",
                "category_type" => "required",
                "venue_name" => "required",
                "sports" => "required",
            ],
            [
                "file.required" => "You have to choose the Venue image!",
                "file.mimes" => "Plese upload jpeg,png,jpg,gif,svg images",
                "category_type.required" => "Category type is required",
                "sports.required" => "Type of Sports is required",
                "venue_name.required" => "Venue name is required",
            ]
        );
        try {
            $image = $request->file("file");
            $imageName = time() . "." . $image->extension();
            $path = public_path("venue");
            if (!File::isDirectory($path)) {
                File::makeDirectory($path, 0777, true, true);
            }
            $sub_foler_arr = ["thumbnail", "images"];
            foreach ($sub_foler_arr as $_data) {
                if (!File::isDirectory($path . "/" . $_data)) {
                    File::makeDirectory($path . "/" . $_data, 0777, true, true);
                }
            }
            $destinationPathThumbnail = public_path("venue/thumbnail");
            $img = Image::make($image->path());
            $img->resize(125, 125, function ($constraint) {
                $constraint->aspectRatio();
            })->save($destinationPathThumbnail . "/" . $imageName);
            $destinationPath = public_path("venue/images");
            $image->move($destinationPath, $imageName);
            $response = Venue::firstOrCreate([
                "name" => $request->venue_name,
                "sports_category" => $request->category_type,
                "sports" => $request->sports,
                "slug" => SlugService::createSlug(
                    Venue::class,
                    "slug",
                    $request->venue_name
                ),
                "image" => "venue/thumbnail/" . $imageName,
            ]);
            if ($response) {
                return response()->json(
                    [
                        "success" => true,
                        "message" => "Venue added successfully",
                    ],
                    200
                );
            }
        } catch (Exception $e) {
            // return response()->json(
            //     [
            //         "success" => false,
            //         "message" => "Some issue.Please contact admin",
            //     ],
            //     204
            // );
            echo $e->getMessage();
        }
    }
    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }
    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        try {
            $Venue = Venue::with("category")->find($id);
            if (!$Venue) {
                return response()->json(
                    ["success" => false, "data" => ""],
                    424
                );
            }
            return response()->json(["success" => true, "data" => $Venue], 200);
        } catch (Exception $e) {
            return response()->json(
                [
                    "success" => false,
                    "message" => "Some issue.Please contact admin",
                ],
                204
            );
        }
    }
    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $this->validate(
            $request,
            [
                "file" => "nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048",
                "category_type" => "required",
                "venue_name" => "required",
                "sports" => "required",
            ],
            [
                "file.mimes" => "Plese upload jpeg,png,jpg,gif,svg images",
                "category_type.required" => "Category type is required",
                "sports.required" => "Type of Sports is required",
                "venue_name.required" => "Venue name is required",
            ]
        );
        try {
            $venues = Venue::find($id);
            //dd($venues);
            $image = $request->file("file");
            if ($image) {
                $imageName = time() . "." . $image->extension();
                $path = public_path("venue");
                if (!File::isDirectory($path)) {
                    File::makeDirectory($path, 0777, true, true);
                }
                $sub_foler_arr = ["thumbnail", "images"];
                foreach ($sub_foler_arr as $_data) {
                    if (!File::isDirectory($path . "/" . $_data)) {
                        File::makeDirectory(
                            $path . "/" . $_data,
                            0777,
                            true,
                            true
                        );
                    }
                }
                $destinationPathThumbnail = public_path("venue/thumbnail");
                $img = Image::make($image->path());
                $img->resize(125, 125, function ($constraint) {
                    $constraint->aspectRatio();
                })->save($destinationPathThumbnail . "/" . $imageName);
                $destinationPath = public_path("venue/images");
                $image->move($destinationPath, $imageName);
                $filename = "venue/thumbnail/" . $imageName;
            } else {
                $filename = $venues->image;
            }
            $venues->name = $request->venue_name;
            $venues->sports_category = $request->category_type;
            $venues->sports = $request->sports;
            $venues->slug = SlugService::createSlug(
                Venue::class,
                "slug",
                $request->venue_name
            );
            $venues->image = $filename;
            $res = $venues->save();
            if ($res) {
                return response()->json(
                    [
                        "success" => true,
                        "message" => "Venue Updated successfully",
                    ],
                    200
                );
            }
        } catch (Exception $e) {
            // return response()->json(
            //     [
            //         "success" => false,
            //         "message" => "Some issue.Please contact admin",
            //     ],
            //     204
            // );
            echo $e->getMessage();
        }
    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
        try {
            $res = Venue::find($id)->delete();
            if (!$res) {
                return response()->json(
                    ["success" => false, "message" => "Venue not deleted !"],
                    424
                );
            }
            return response()->json(
                [
                    "success" => true,
                    "message" => "Venue was deleted successfully!",
                ],
                200
            );
        } catch (Exception $e) {
            return response()->json(
                [
                    "success" => false,
                    "message" => "Some issue.Please contact admin",
                ],
                204
            );
        }
    }
    public function timeSchedule($id)
    {
        $court_id = $id;
        $player_type = PlayerType::all();
        //    $data = Days::with(array('allSlots' => function($query) use ($court_id) {
        //     $query->where('court_id', $court_id);
        //    }))->get();
        //   dd($data[0]->allSlots->id);
        return view(
            "admin.court-time-schedule",
            compact("court_id", "player_type")
        );
    }
    public function loadTimings($court_id)
    {
        $data = Days::with([
            "allSlots" => function ($query) use ($court_id) {
                $query->where("court_id", $court_id);
            },
        ])->get();
        return view(
            "admin.includes.court_timings",
            compact("data", "court_id")
        );
    }
    public function loadDayTimings($slot_id, $day_id)
    {
        $response = SlotTime::where([
            "slot_id" => $slot_id,
            "day_id" => $day_id,
        ])->get();
        //echo json_encode($response);die;
        $custom = CutomSlots::where(["id" => $slot_id])->first();
        //$date = $custom->date;
        $date = '';
        return view(
            "admin.includes.court_day_time_for_popup",
            compact("response", "slot_id", "day_id", "date")
        );
    }

    public function saveTimeSchedule(Request $request, $id)
    {
        $this->validate(
            $request,
            [
                "start_time" => [
                    "array",
                    new WeeklySlotTimeSave($request->end_time, $id),
                ],
                "start_time.*" => "distinct|required_with:status.*,1",
                "end_time" => "array",
                "end_time.*" => "distinct",
                "break_time" => "array",
                "break_time.*" => "distinct",
                "hours_per_slot" => "array",
                "hours_per_slot.*.*" => "required_with:status.*.*,1",
                "price" => "array",
                "price.*" => "distinct|required_with:status.*,1",
            ],
            [
                "start_time.required" => "Please enter start time",
                "start_time.*.required" => "Some fields start time empty",
                "end_time.required" => "Please enter end time",
                "end_time.*.required" => "Some fields end time empty",
            ]
        );

        try {
            $days = Days::all();
            foreach ($days as $_data) {
                if (isset($request->status[$id][$_data->id])) {
                    $coaching_data = Coaching::with("Timings")
                        ->where(["court_id" => $id])
                        ->whereJsonContains("weekid", [(string) $_data->id])
                        ->first();
                    if (isset($coaching_data->timings)) {
                        $collection = collect($coaching_data->timings);
                    } else {
                        $collection = collect();
                    }

                    // $data=array_column($coaching_data['timings'],'start_time');
                    // dd($coaching_data['timings']);
                    $response = Slots::updateOrCreate(
                        ["day_id" => $_data->id, "court_id" => $id],
                        [
                            "day_id" => $_data->id,
                            "start_time" =>
                                $request->start_time[$id][$_data->id],
                            "end_time" => $request->end_time[$id][$_data->id],
                            "slot_break_time" =>
                                $request->break_time[$id][$_data->id],
                            "slot_duration" =>
                                $request->hours_per_slot[$id][$_data->id],
                            "price_per_slot" =>
                                $request->price[$id][$_data->id],
                            "time_off" => isset(
                                $request->status[$id][$_data->id]
                            )
                                ? 1
                                : 0,
                            "court_id" => $id,
                        ]
                    );
                    //echo json_encode( $response );
                    if ($response) {
                        if (
                            isset($request->reset[$id][$_data->id]) &&
                            $request->reset[$id][$_data->id] == 1
                        ) {
                            SlotTime::where([
                                "day_id" => $_data->id,
                                "slot_id" => $response->id,
                            ])->delete();
                            $Duration =
                                $request->hours_per_slot[$id][$_data->id] * 60;
                            $break = $request->break_time[$id][$_data->id];
                            $retrun_data = $this->SplitTime(
                                $request->start_time[$id][$_data->id],
                                $request->end_time[$id][$_data->id],
                                $Duration,
                                $break,
                                $_data->id,
                                $response->id,
                                $request->price[$id][$_data->id],
                                $id,
                                clone $collection
                            );
                            //echo json_encode($retrun_data);
                            SlotTime::insert($retrun_data);
                        } else {
                            //echo "hi vicky";
                        }
                    }
                } else {
                    $response = Slots::updateOrCreate(
                        ["day_id" => $_data->id, "court_id" => $id],
                        [
                            "day_id" => $_data->id,
                            "time_off" => 0,
                            "court_id" => $id,
                        ]
                    );
                }
            }

            return response()->json(
                [
                    "success" => true,
                    "message" => "Slot time updated successfully!..",
                ],
                200
            );
        } catch (Exception $e) {
            echo $e->getMessage();
        }
    }

    public function SplitTime(
        $StartTime,
        $EndTime,
        $Duration = "60",
        $break,
        $day_id,
        $slot_id,
        $price,
        $cid,
        $collection = null
    ) {
        $ReturnArray = []; // Define output
        $StartTime = strtotime($StartTime); //Get Timestamp
        $EndTime = strtotime($EndTime); //Get Timestamp
        $AddMins = $Duration * 60;
        while ($StartTime <= $EndTime) {
            //Run loop

            $data = [];
            // if($collection->Where('end_time', '>=', date("H:i", $StartTime + $AddMins))->Where('start_time', '<=', date("H:i", $StartTime))){
            //     echo "<pre>";print_r($collection);
            // }
            if (
                $collection != null &&
                ($data1 = $collection
                    ->Where(
                        "end_time",
                        ">=",
                        date("H:i", $StartTime + $AddMins)
                    )
                    ->Where("start_time", "<=", date("H:i", $StartTime)))
            ) {
                $data = json_decode($data1, true);
                //echo json_encode($data1);
            }
            if (!$data) {
                //  echo (date("H:i", $StartTime));
                $ReturnArray[] = [
                    "slot_id" => $slot_id,
                    "day_id" => $day_id,
                    "price" => $price,
                    "time_from" => date("H:i", $StartTime),
                    "time_to" => date("H:i", $StartTime + $AddMins),
                    "golden_hour" => 0,
                    "status" => 1,
                    "court_id" => $cid,
                ];
            }
            $StartTime += $AddMins + $break * 60;
        }

        return $ReturnArray;
    }
    public function updateTimings(Request $request)
    {
        $slot_id = $request->slot_id;
        $day_id = $request->day_id;
        $days = Days::all();
        for ($i = 0; $i < count($request->time_from); $i++) {
            $temp[] = [
                "price" => $request->price[$i],
                "status" => $request->status[$i] ? 1 : 0,
                "slot_id" => $slot_id,
                "day_id" => $day_id,
                "time_from" => $request->time_from[$i],
                "time_to" => $request->time_to[$i],
                "golden_hour" => isset($request->golden_hour[$i]) ? $request->golden_hour[$i]: 0 ,
                "golden_price"=> isset($request->golden_hour[$i]) ? $request->golden_price[$i]: $request->price[$i] , 
                "court_id" => $request->court_id,
            ];
        }
        SlotTime::where(["day_id" => $day_id, "slot_id" => $slot_id])->delete();
        SlotTime::insert($temp);
        return response()->json(
            ["success" => true, "message" => "Slot time updated successfully "],
            200
        );
    }
    public function timeCustomSchedule($id)
    {
        $data = CutomSlots::orderBy("id", "DESC")->get();
        return Datatables::of($data)
            ->addIndexColumn()
            ->addColumn("action", function ($row) {
                $btn =
                    '
                <button type="button" class="btn btn-icon btn-active-light-primary w-30px h-30px ms-auto show_custom_time" data-id="' .
                    $row->id .
                    '" data-bs-toggle="tooltip" data-bs-custom-class="tooltip-inverse" data-bs-placement="top" title="Schedule time">
                <span class="svg-icon svg-icon-3">
                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><polyline points="12 6 12 12 15 15"></polyline></svg>
                </span>
                </button>
                <button type="button" class="btn btn-icon btn-active-light-primary w-30px h-30px ms-auto edit_custom_time" data-id="' .
                    $row->id .
                    '">
                <span class="svg-icon svg-icon-3">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path opacity="0.3" d="M21.4 8.35303L19.241 10.511L13.485 4.755L15.643 2.59595C16.0248 2.21423 16.5426 1.99988 17.0825 1.99988C17.6224 1.99988 18.1402 2.21423 18.522 2.59595L21.4 5.474C21.7817 5.85581 21.9962 6.37355 21.9962 6.91345C21.9962 7.45335 21.7817 7.97122 21.4 8.35303ZM3.68699 21.932L9.88699 19.865L4.13099 14.109L2.06399 20.309C1.98815 20.5354 1.97703 20.7787 2.03189 21.0111C2.08674 21.2436 2.2054 21.4561 2.37449 21.6248C2.54359 21.7934 2.75641 21.9115 2.989 21.9658C3.22158 22.0201 3.4647 22.0084 3.69099 21.932H3.68699Z" fill="currentColor"></path>
                    <path d="M5.574 21.3L3.692 21.928C3.46591 22.0032 3.22334 22.0141 2.99144 21.9594C2.75954 21.9046 2.54744 21.7864 2.3789 21.6179C2.21036 21.4495 2.09202 21.2375 2.03711 21.0056C1.9822 20.7737 1.99289 20.5312 2.06799 20.3051L2.696 18.422L5.574 21.3ZM4.13499 14.105L9.891 19.861L19.245 10.507L13.489 4.75098L4.13499 14.105Z" fill="currentColor"></path>
                </svg>
                </span>
                </button>
                <button type="button" class="btn btn-icon btn-active-light-primary w-30px h-30px ms-auto delete_custom_time" data-id="' .
                    $row->id .
                    '" id="kt_users_delete_two_step">
                <span class="svg-icon svg-icon-3">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M5 9C5 8.44772 5.44772 8 6 8H18C18.5523 8 19 8.44772 19 9V18C19 19.6569 17.6569 21 16 21H8C6.34315 21 5 19.6569 5 18V9Z" fill="currentColor"></path>
                    <path opacity="0.5" d="M5 5C5 4.44772 5.44772 4 6 4H18C18.5523 4 19 4.44772 19 5V5C19 5.55228 18.5523 6 18 6H6C5.44772 6 5 5.55228 5 5V5Z" fill="currentColor"></path>
                    <path opacity="0.5" d="M9 4C9 3.44772 9.44772 3 10 3H14C14.5523 3 15 3.44772 15 4V4H9V4Z" fill="currentColor"></path>
                </svg>
                </span>
                </button>';
                return $btn;
            })
            ->editColumn("time_off", function ($row) {
                if ($row->time_off == 0) {
                    return "Leave";
                } else {
                    return "Working";
                }
            })
            ->rawColumns(["action", "time_off"])
            ->make(true);
    }
    public function timeSaveCustomSchedule(Request $request, $id)
    {
        $this->validate(
            $request,
            [
                "day_date" =>
                    "required|unique:custom_slots,date,NULL,NULL,court_id," .
                    $id,
            ],
            [
                "day_date.required" => "Please select day id",
                "day_date.unique" => "Alreay assign schedule for this date",
            ]
        );
        if (!$this->alreadyBookSlot($request->day_date)) {
            return response()->json(
                [
                    "success" => false,
                    "message" =>
                        "Already customer booked slot in this date , So cant add this date",
                ],
                200
            );
        }
        if (
            isset($request->day_working_day) &&
            $request->day_working_day == "on"
        ) {
            $this->validate(
                $request,
                [
                    "day_start_time" => "required",
                    "day_end_time" => "required",
                    "day_break_time" => "required",
                    "day_slot_duration" => "required",
                    "day_slot_price" => "required",
                    "day_working_day" => "required",
                ],
                [
                    "day_date.required" => "Please select day id",
                    "day_date.unique" => "Alreay assign schedule for this date",
                ]
            );
            $response = CutomSlots::updateOrCreate(
                ["date" => $request->day_date, "court_id" => $id],
                [
                    "start_time" => $request->day_start_time,
                    "end_time" => $request->day_end_time,
                    "slot_break_time" => $request->day_break_time,
                    "slot_duration" => $request->day_slot_duration,
                    "price_per_slot" => $request->day_slot_price,
                    "time_off" => isset($request->day_working_day) ? 1 : 0,
                    "court_id" => $id,
                ]
            );
            if ($response) {
                if (
                    isset($request->day_working_day) &&
                    $request->day_working_day == "on"
                ) {
                    CutomSlotsTiming::where([
                        "day_date" => $request->day_date,
                        "court_id" => $id,
                    ])->delete();
                    $Duration = $request->day_slot_duration * 60;
                    $break = $request->day_break_time;
                    $retrun_data = $this->CustomSplitTime(
                        $request->day_start_time,
                        $request->day_end_time,
                        $Duration,
                        $break,
                        $request->day_date,
                        $response->id,
                        $request->day_slot_price,
                        $id
                    );
                    CutomSlotsTiming::insert($retrun_data);
                }
            }
        } else {
            $response = CutomSlots::updateOrCreate(
                ["date" => $request->day_date, "court_id" => $id],
                [
                    "date" => $request->day_date,
                    "time_off" => 0,
                    "court_id" => $id,
                ]
            );
        }
        return response()->json(
            ["success" => true, "message" => "Slot Updated successfully"],
            200
        );
    }

    public function alreadyBookSlot($date)
    {
        $count = Booking::where("date", $date)->count();
        if ($count == 0) {
            return true;
        }
        return false;
    }
    public function CustomSplitTime(
        $StartTime,
        $EndTime,
        $Duration = "60",
        $break,
        $day_id,
        $slot_id,
        $price,
        $cid
    ) {
        $ReturnArray = []; // Define output
        $StartTime = strtotime($StartTime); //Get Timestamp
        $EndTime = strtotime($EndTime); //Get Timestamp
        $AddMins = $Duration * 60;
        while ($StartTime <= $EndTime) {
            //Run loop

            //Endtime check
            $ReturnArray[] = [
                "slot_id" => $slot_id,
                "day_date" => $day_id,
                "price" => $price,
                "time_from" => date("H:i", $StartTime),
                "time_to" => date("H:i", $StartTime + $AddMins),
                "golden_hour" => 0,
                "status" => 1,
                "court_id" => $cid,
            ];
            $StartTime += $AddMins + $break * 60;
        }
        return $ReturnArray;
    }

    public function timeCustomScheduleEdit($id)
    {
        try {
            $CutomSlots = CutomSlots::find($id);
            if (!$CutomSlots) {
                return response()->json(
                    ["success" => false, "data" => ""],
                    424
                );
            }
            return response()->json(
                ["success" => true, "data" => $CutomSlots],
                200
            );
        } catch (Exception $e) {
            return response()->json(
                [
                    "success" => false,
                    "message" => "Some issue.Please contact admin",
                ],
                204
            );
        }
    }

    public function timeCustomScheduleUpdate(Request $request, $id)
    {
        $this->validate(
            $request,
            ["day_date" => "required|unique:custom_slots,date," . $id],
            [
                "day_date.required" => "Please select day id",
                "day_date.unique" => "Alreay assign schedule for this date",
            ]
        );
        if (
            isset($request->day_working_day) &&
            $request->day_working_day == "on"
        ) {
            $this->validate(
                $request,
                [
                    "day_start_time" => "required",
                    "day_end_time" => "required",
                    "day_break_time" => "required",
                    "day_slot_duration" => "required",
                    "day_slot_price" => "required",
                    "day_working_day" => "required",
                ],
                [
                    "day_date.required" => "Please select day id",
                    "day_date.unique" => "Alreay assign schedule for this date",
                ]
            );

            $response = CutomSlots::updateOrCreate(
                ["date" => $request->day_date, "id" => $id],
                [
                    "start_time" => $request->day_start_time,
                    "end_time" => $request->day_end_time,
                    "slot_break_time" => $request->day_break_time,
                    "slot_duration" => $request->day_slot_duration,
                    "price_per_slot" => $request->day_slot_price,
                    "time_off" => isset($request->day_working_day) ? 1 : 0,
                    "court_id" => $id,
                ]
            );
            if ($response) {
                if (
                    isset($request->day_working_day) &&
                    $request->day_working_day == "on"
                ) {
                    CutomSlotsTiming::where([
                        "day_date" => $request->day_date,
                        "slot_id" => $id,
                    ])->delete();
                    $Duration = $request->day_slot_duration * 60;
                    $break = $request->day_break_time;
                    $retrun_data = $this->CustomSplitTime(
                        $request->day_start_time,
                        $request->day_end_time,
                        $Duration,
                        $break,
                        $request->day_date,
                        $response->id,
                        $request->day_slot_price,
                        $id
                    );
                    CutomSlotsTiming::insert($retrun_data);
                }
            }
        } else {
            $response = CutomSlots::updateOrCreate(
                ["date" => $request->day_date, "id" => $id],
                [
                    "date" => $request->day_date,
                    "time_off" => 0,
                    "court_id" => $id,
                ]
            );
        }

        return response()->json(
            ["success" => true, "message" => "Slot Updated successfully"],
            200
        );
    }

    public function timeCustomScheduleDelete($id)
    {
        try {
            $res = CutomSlots::find($id)->delete();
            $res1 = CutomSlotsTiming::where("slot_id", $id)->delete();
            if (!$res || !$res1) {
                return response()->json(
                    [
                        "success" => false,
                        "message" => "Slot Can't be deleted !",
                    ],
                    424
                );
            }
            return response()->json(
                [
                    "success" => true,
                    "message" => "Slot was deleted successfully!",
                ],
                200
            );
        } catch (Exception $e) {
            return response()->json(
                [
                    "success" => false,
                    "message" => "Some issue.Please contact admin",
                ],
                204
            );
        }
    }

    public function timeCustomSlotTimingEdit($slot_id)
    {
        $custom = CutomSlots::where(["id" => $slot_id])->first();
        $date = $custom->date;
        $response = CutomSlotsTiming::where(["slot_id" => $slot_id])->get();
        $day_id = null;
        return view(
            "admin.includes.court_day_time_for_popup",
            compact("response", "slot_id", "day_id", "date")
        );
    }

    public function timeCustomSlotTimingUpdate(Request $request, $id)
    {
        $this->validate(
            $request,
            [
                "time_from" => [
                    "required",
                    "array",
                    new CouchingTimeSave($request->time_to),
                ],
                "time_from.*" => "required|string|distinct",
                "time_to" => "required|array",
                "time_to.*" => "required|string|distinct",
                "price" => "required|array",
                "price.*" => "required",
             
            ],
            [
                "time_from.required" => "Please enter start time",
                "time_from.*.required" => "Some fields start time empty",
                "time_to.required" => "Please enter end time",
                "time_to.*.required" => "Some fields end time empty",
                "price.required" => "Please enter amount",
                "price.*.required" => "Amount Some fields empty",
              
            ]
        );

        $slot_id = $request->slot_id;
        $day_id = $request->day_id;
        $date = $request->date;
        if (!$this->alreadyBookSlot($date)) {
            return response()->json(
                [
                    "success" => false,
                    "message" =>
                        "Already customer booked slot in this date , So cant add this date",
                ],
                200
            );
        }
        $slots = CutomSlotsTiming::where(["slot_id" => $slot_id])->delete();
       
        for ($i = 0; $i < count($request->time_from); $i++) {
             $ReturnArray[] = [
                "slot_id" => $slot_id,
                "day_date" => $day_id,
                "price" => $request->price[$i],
                "time_from" => date("H:i", strtotime($request->time_from[$i])),
                "time_to" => date("H:i", strtotime($request->time_to[$i])),
                "golden_hour" => 0,
                "status" => 1,
                "court_id" => $request->court_id,
            ];
        }
       
        $data= CutomSlotsTiming::insert($ReturnArray);
        if($data){
            return response()->json(
                ["success" => true, "message" => "Slot time updated successfully "],
                200
            );
        }
        
    }

    public function getCouchingSlots($id)
    {
        $data = Coaching::where("court_id", $id)
            ->orderBy("id", "DESC")
            ->get();

        return Datatables::of($data)
            ->addIndexColumn()
            ->addColumn("action", function ($row) {
                $btn =
                    '
                <button type="button" class="btn btn-icon btn-active-light-primary w-30px h-30px ms-auto show_coaching_time" data-id="' .
                    $row->id .
                    '" data-bs-toggle="tooltip" data-bs-custom-class="tooltip-inverse" data-bs-placement="top" title="Schedule time">
                <span class="svg-icon svg-icon-3">
                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"></circle><polyline points="12 6 12 12 15 15"></polyline></svg>
                </span>
                </button>
                <button type="button" class="btn btn-icon btn-active-light-primary w-30px h-30px ms-auto edit_coaching_time" data-id="' .
                    $row->id .
                    '">
                <span class="svg-icon svg-icon-3">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path opacity="0.3" d="M21.4 8.35303L19.241 10.511L13.485 4.755L15.643 2.59595C16.0248 2.21423 16.5426 1.99988 17.0825 1.99988C17.6224 1.99988 18.1402 2.21423 18.522 2.59595L21.4 5.474C21.7817 5.85581 21.9962 6.37355 21.9962 6.91345C21.9962 7.45335 21.7817 7.97122 21.4 8.35303ZM3.68699 21.932L9.88699 19.865L4.13099 14.109L2.06399 20.309C1.98815 20.5354 1.97703 20.7787 2.03189 21.0111C2.08674 21.2436 2.2054 21.4561 2.37449 21.6248C2.54359 21.7934 2.75641 21.9115 2.989 21.9658C3.22158 22.0201 3.4647 22.0084 3.69099 21.932H3.68699Z" fill="currentColor"></path>
                    <path d="M5.574 21.3L3.692 21.928C3.46591 22.0032 3.22334 22.0141 2.99144 21.9594C2.75954 21.9046 2.54744 21.7864 2.3789 21.6179C2.21036 21.4495 2.09202 21.2375 2.03711 21.0056C1.9822 20.7737 1.99289 20.5312 2.06799 20.3051L2.696 18.422L5.574 21.3ZM4.13499 14.105L9.891 19.861L19.245 10.507L13.489 4.75098L4.13499 14.105Z" fill="currentColor"></path>
                </svg>
                </span>
                </button>
                <button type="button" class="btn btn-icon btn-active-light-primary w-30px h-30px ms-auto delete_coaching_time" data-id="' .
                    $row->id .
                    '" id="kt_users_delete_two_step">
                <span class="svg-icon svg-icon-3">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M5 9C5 8.44772 5.44772 8 6 8H18C18.5523 8 19 8.44772 19 9V18C19 19.6569 17.6569 21 16 21H8C6.34315 21 5 19.6569 5 18V9Z" fill="currentColor"></path>
                    <path opacity="0.5" d="M5 5C5 4.44772 5.44772 4 6 4H18C18.5523 4 19 4.44772 19 5V5C19 5.55228 18.5523 6 18 6H6C5.44772 6 5 5.55228 5 5V5Z" fill="currentColor"></path>
                    <path opacity="0.5" d="M9 4C9 3.44772 9.44772 3 10 3H14C14.5523 3 15 3.44772 15 4V4H9V4Z" fill="currentColor"></path>
                </svg>
                </span>
                </button>';
                return $btn;
            })
            ->editColumn("weekid", function ($row) {
                $rray = $row->weekid;
                $days = Days::select("name")
                    ->whereIn("id", $rray)
                    ->get()
                    ->toArray();
                return array_column($days, "name");
            })
            ->rawColumns(["action", "time_off"])
            ->make(true);
    }

    public function saveCouchingSlots(Request $request, $id)
    {
        $this->validate(
            $request,
            [
                "group_name" =>
                    "required|unique:coaching_schedule,group,NULL,NULL,court_id," .
                    $id,
                "days" => "required",
            ],
            [
                "group_name.required" => "Please enter Group Name",
                "days.required" => "Please Select days",
            ]
        );

        $days = json_decode($request->days, true);
        $w_days = array_column($days, "code");
        $day_exits = Coaching::whereJsonContains("weekid", $w_days)
            ->where(["court_id" => $id])
            ->get();
        // echo json_encode($day_exits);
        if (count($day_exits) > 0) {
            return response()->json(
                [
                    "success" => false,
                    "message" => "Already some weekday assigned other session",
                ],
                200
            );
        }

        $flight = Coaching::firstOrCreate([
            "group" => $request->group_name,
            "weekid" => $w_days,
            "court_id" => $id,
        ]);
        if ($flight) {
            return response()->json(
                ["success" => true, "message" => "Couching successfully"],
                200
            );
        } else {
            return response()->json(
                ["success" => false, "message" => "Couching not updated"],
                200
            );
        }
    }

    public function LoadCouchingSlotsTiming($id)
    {
        $slots = CoachingTimings::where("cs_id", $id)->get();
        $player_type = PlayerType::all();
        return view(
            "admin.includes.couching_time",
            compact("id", "slots", "player_type")
        );
    }

    public function LoadCouchingsaveSlotsTiming(Request $request, $id)
    {
        $this->validate(
            $request,
            [
                "start_time" => [
                    "required",
                    "array",
                    new CouchingTimeSave($request->end_time),
                ],
                "start_time.*" => "required|string|distinct",
                "end_time" => "required|array",
                "end_time.*" => "required|string|distinct",
                "break_time" => "required|array",
                "break_time.*" => "required",
                "customer_type" => "required|array",
                "customer_type.*" => "required",
                "customer_level" => "required|array",
                "customer_level.*" => "required",
                "duration" => "required|array",
                "duration.*" => "required",
                "amount" => "required|array",
                "amount.*" => "required",
                "max_join" => "required|array",
                "max_join.*" => "required",
            ],
            [
                "start_time.required" => "Please enter start time",
                "start_time.*.required" => "Some fields start time empty",
                "end_time.required" => "Please enter end time",
                "end_time.*.required" => "Some fields end time empty",
                "break_time.required" => "Please enter Break time",
                "break_time.*.required" => "Some fields break time empty",
                "customer_type.required" => "Please select customer type",
                "customer_type.*.required" => "Customer Type Some fields empty",
                "customer_level.required" => "Please select customer Level",
                "customer_level.*.required" =>
                    "Customer Level Some fields empty",
                "duration.required" => "Please select duration",
                "duration.*.required" => "Duration some fields empty",
                "amount.required" => "Please enter amount",
                "amount.*.required" => "Amount Some fields empty",
                "max_join.required" => "Please enter Maximum join members",
                "max_join.*.required" => "Maximum join Some fields empty",
            ]
        );

        for ($i = 0; $i < count($request->start_time); $i++) {
            $temp[] = [
                "start_time" => $request->start_time[$i],
                "end_time" => $request->end_time[$i],
                "break_time" => $request->break_time[$i],
                "customer_type" => $request->customer_type[$i],
                "customer_level" => $request->customer_level[$i],
                "duration" => $request->duration[$i],
                "amount" => $request->amount[$i],
                "max_join" => $request->max_join[$i],
                "cs_id" => $request->couching_slot,
                "court_id" => $id,
                "status" => $request->status[$i] ? 1 : 0,
            ];
        }

        $res = CoachingTimings::where(
            "cs_id",
            $request->couching_slot
        )->delete();

        $res = CoachingTimings::insert($temp);
        if ($res) {
            return response()->json(
                [
                    "success" => true,
                    "message" => "Couching time updated successfully!",
                ],
                200
            );
        }

        return response()->json(
            [
                "success" => false,
                "message" => "Couching time not updated successfully!",
            ],
            404
        );
    }

    public function LoadCouchingSlotsTimingTr()
    {
        $player_type = PlayerType::all();
        return view(
            "admin.includes.partial.couchtime_single_row",
            compact("player_type")
        );
    }

    public function quickLoadCouchingSlotsTimingTr(Request $request)
    {
        $this->validate(
            $request,
            [
                "start_time" => "required",
                "end_time" => "required",
                "hours" => "required",
                "price" => "required",
                "breaktime" => "required",
                "customer_type" => "required",
            ],
            [
                "start_time.required" => "Start time is required",
                "end_time.required" => "End time is required",
                "hours.required" => "Hours is required",
                "price.required" => "Price is required",
                "breaktime.required" => "Breaktime is required",
                "customer_type.required" => "Customer type is required",
            ]
        );
        $ReturnArray = []; // Define output
        $StartTime = strtotime($request->start_time); //Get Timestamp
        $EndTime = strtotime($request->end_time); //Get Timestamp
        $AddMins = $request->hours * 60 * 60;
        $tr_tag = "";
        $player_type = PlayerType::all();
        while ($StartTime <= $EndTime) {
            //Run loop

            //Endtime check
            $ReturnArray = [
                "price" => $request->price,
                "time_from" => date("H:i", $StartTime),
                "time_to" => date("H:i", $StartTime + $AddMins),
                "hours" => $request->hours,
                "break" => $request->breaktime,
                "status" => $request->status ? 1 : 0,
                "max_join" => $request->max_join,
                "customer_type" => $request->customer_type,
            ];
            //echo json_encode( $ReturnArray);
            $tr_tag .= view(
                "admin.includes.partial.initial_couchtime",
                compact("ReturnArray", "player_type")
            )->render();
            $StartTime += $AddMins + $request->breaktime * 60;
        }
        return $tr_tag;
    }

    public function getCouchingSession($id)
    {
        $data = Coaching::find($id);
        // echo json_encode($data);
        if (!$data) {
            return response()->json(["success" => false, "data" => ""], 424);
        }
        return response()->json(["success" => true, "data" => $data], 200);
    }

    public function updateCouchingSlots($cid, $id, Request $request)
    {
        $this->validate(
            $request,
            [
                "group_name" =>
                    "required|unique:coaching_schedule,group," .
                    $id .
                    ",id,court_id," .
                    $cid,
                "days" => "required",
                "coaching_session_id" => "required",
            ],
            [
                "group_name.required" => "Please enter Group Name",
                "days.required" => "Please Select days",
            ]
        );

        $days = json_decode($request->days, true);
        $w_days = array_column($days, "code");
        $day_exits = Coaching::whereJsonContains("weekid", $w_days)
            ->where(["court_id" => $id])
            ->get();
        // echo json_encode($day_exits);
        if (count($day_exits) > 0) {
            return response()->json(
                [
                    "success" => false,
                    "message" => "Already some weekday assigned other session",
                ],
                200
            );
        }

        $flight = Coaching::where("id", $request->id)->update([
            "group" => $request->group_name,
            "weekid" => $w_days,
            "court_id" => $cid,
        ]);
        if ($flight) {
            return response()->json(
                ["success" => true, "message" => "Couching successfully"],
                200
            );
        } else {
            return response()->json(
                ["success" => false, "message" => "Couching not updated"],
                200
            );
        }
    }

    public function LoadCustomSlotsTimingTr()
    {
        return view("admin.includes.partial.slottime_single_row");
    }
}
